package httpclient

import (
	"bytes"
	"io"
	"net/http"
	"time"
)

type Client struct {
	client     *http.Client
	maxRetries int
	retryDelay time.Duration
}

func NewClient(timeout time.Duration, maxRetries int, retryDelay time.Duration) *Client {
	return &Client{
		client: &http.Client{
			Timeout: timeout,
		},
		maxRetries: maxRetries,
		retryDelay: retryDelay,
	}
}

func (c *Client) Get(url string, headers map[string]string) (*http.Response, error) {
	req, err := http.NewRequest("GET", url, nil)
	if err != nil {
		return nil, err
	}
	return c.doWithRetry(req, headers)
}

func (c *Client) Post(url string, body []byte, headers map[string]string) (*http.Response, error) {
	req, err := http.NewRequest("POST", url, bytes.NewBuffer(body))
	if err != nil {
		return nil, err
	}
	return c.doWithRetry(req, headers)
}

func (c *Client) doWithRetry(req *http.Request, headers map[string]string) (*http.Response, error) {
	for k, v := range headers {
		req.Header.Set(k, v)
	}
	
	var lastErr error
	for i := 0; i <= c.maxRetries; i++ {
		resp, err := c.client.Do(req)
		if err == nil && resp.StatusCode < 500 {
			return resp, nil
		}
		if resp != nil {
			resp.Body.Close()
		}
		lastErr = err
		if i < c.maxRetries {
			time.Sleep(c.retryDelay)
		}
	}
	return nil, lastErr
}