import csv
from typing import List, Dict, Callable, Optional
from pathlib import Path


class CSVProcessor:
    def __init__(self, filepath: str, delimiter: str = '{{delimiter}}'):
        self.filepath = Path(filepath)
        self.delimiter = delimiter
        self.data: List[Dict[str, str]] = []
    
    def read(self) -> 'CSVProcessor':
        """Read CSV file into memory."""
        with open(self.filepath, 'r', encoding='utf-8') as f:
            reader = csv.DictReader(f, delimiter=self.delimiter)
            self.data = list(reader)
        return self
    
    def filter(self, condition: Callable[[Dict[str, str]], bool]) -> 'CSVProcessor':
        """Filter rows based on condition function."""
        self.data = [row for row in self.data if condition(row)]
        return self
    
    def transform(self, column: str, func: Callable[[str], str]) -> 'CSVProcessor':
        """Transform values in a column."""
        for row in self.data:
            if column in row:
                row[column] = func(row[column])
        return self
    
    def get_data(self) -> List[Dict[str, str]]:
        """Return processed data."""
        return self.data
    
    def write(self, output_path: str) -> None:
        """Write processed data to CSV file."""
        if not self.data:
            return
        
        with open(output_path, 'w', newline='', encoding='utf-8') as f:
            writer = csv.DictWriter(f, fieldnames=self.data[0].keys(), delimiter=self.delimiter)
            writer.writeheader()
            writer.writerows(self.data)